﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using Unity.Mathematics;
using UnityEngine;
using UnityEngine.Tilemaps;
using UnityEngine.XR;

//BuildingsManager is a singleton class responsible for spawning and handling all buildings in the game
public class BuildingsManager : MonoBehaviour
{
    //BuildingTileInfo is a small container class that contains all info about a building tile
    public class BuildingTileInfo
    {
        public Vector3Int Position { get; private set; }
        public bool IsAdjacentToRoad { get; private set; }
        public bool IsInternal { get; private set; }
        public int ChunkX { get; private set; }
        public int ChunkY { get; private set; }

        public bool IsOccupied { get; private set; }
        public GameObject BuildingGameObject { get; private set; }

        public BuildingTileInfo(Vector3Int position, bool isAdjacentToRoad, bool isInternal, int chunkX, int chunkY)
        {
            Position = position;
            IsAdjacentToRoad = isAdjacentToRoad;
            IsInternal = isInternal;
            ChunkX = chunkX;
            ChunkY = chunkY;
        }

        public void SetOccupied(GameObject buildingGameObject)
        {
            IsOccupied = buildingGameObject != null;
            BuildingGameObject = buildingGameObject;
        }
    }

    //Singleton
    private static BuildingsManager _Instance;

    public static BuildingsManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<BuildingsManager>();
            }

            return _Instance;
        }
    }

    public BuildingsInfo BuildingsInfo { get; private set; }

    //Boundaries of buildings in the city
    public Vector3Int BuildingAreaTopLeftTilePosition { get; private set; }
    public Vector3Int BuildingAreaBottomLeftTilePosition { get; private set; }
    public Vector3Int BuildingAreaTopRightTilePosition { get; private set; }
    public Vector3Int BuildingAreaBottomRightTilePosition { get; private set; }
    public Vector2Int BuildingAreaSize { get; private set; }

    //Positions of buildings in the city
    public Dictionary<Vector3Int, BuildingTileInfo> RoadBuildings { get; private set; } = new Dictionary<Vector3Int, BuildingTileInfo>();
    public Dictionary<Vector3Int, BuildingTileInfo> Garages { get; private set; } = new Dictionary<Vector3Int, BuildingTileInfo>();

    private Tile _TransparentTile;
    private Dictionary<Vector3Int, BuildingTileInfo> _BuildingTiles;
    private List<int> _UsedKeyIndexes = new List<int>();
    private PerlinNoiseGenerator _PerlinNoise;

    //Distance from city centre for building types
    private float _SkyscraperDistance;
    private float _CommercialDistance;
    private float _NoiseZoneDistance;

    /// <summary>
    /// Initializes the BuildingsManager by parsing the buildings info
    /// </summary>
    /// <returns>Was the initialization successful?</returns>
    public bool Initialize()
    {
        try
        {
            _TransparentTile = Resources.Load("Palette Tiles/transparent") as Tile;
            return ParseBuildingsInfo();
        }

        catch
        {
            return false;
        }
    }

    /// <summary>
    /// Parses the BuildingsInfo script for buildings
    /// </summary>
    /// <returns>Did the parse succeed?</returns>
    private bool ParseBuildingsInfo()
    {
        try
        {
            BuildingsInfo = new BuildingsInfo();
            return BuildingsInfo.Parse("Buildings/BuildingsInfo");
        }

        catch (Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }

    /// <summary>
    /// Adds a new building tile info for the passed tile position
    /// </summary>
    /// <param name="tilePos">The tile position</param>
    /// <param name="isInternal">Is the tile on the inside of the city?</param>
    /// <returns>Was the tile added successfully?</returns>
    private bool AddNewBuildingTile(Vector3Int tilePos, bool isInternal)
    {
        try
        {
            //Set the tile as transparent, update its name to be valid
            Tile thisTransparentTile = Instantiate(_TransparentTile) as Tile;
            thisTransparentTile.name = Constants.ValidBuildingPlacementTileID;
            GameManager.Instance.BuildingsTilemap.SetTile(tilePos, thisTransparentTile);

            TileBase buildingTile = GameManager.Instance.BuildingsTilemap.GetTile(tilePos);

            buildingTile.name = Constants.ValidBuildingPlacementTileID;

            //Get a list of possible rotations for the placed building
            List<float> possibleRotations = new List<float>();

            //Get the possible rotation values based on the neighbouring roads
            if (tilePos.x > 0)
            {
                TileBase leftNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(tilePos.x - 1, tilePos.y, tilePos.z));

                if (leftNeighbour != null && ((Tile)leftNeighbour).name.Contains("Road"))
                {
                    possibleRotations.Add(90.0f);
                }
            }

            if (tilePos.x < GameManager.Instance.RoadsTilemap.size.x)
            {
                TileBase rightNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(tilePos.x + 1, tilePos.y, tilePos.z));

                if (rightNeighbour != null && ((Tile)rightNeighbour).name.Contains("Road"))
                {
                    possibleRotations.Add(270.0f);
                }
            }

            if (tilePos.y > 0)
            {
                TileBase topNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(tilePos.x, tilePos.y + 1, tilePos.z));

                if (topNeighbour != null && ((Tile)topNeighbour).name.Contains("Road"))
                {
                    possibleRotations.Add(180.0f);
                }
            }

            if (tilePos.y < GameManager.Instance.RoadsTilemap.size.y)
            {
                TileBase bottomNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(tilePos.x, tilePos.y - 1, tilePos.z));

                if (bottomNeighbour != null && ((Tile)bottomNeighbour).name.Contains("Road"))
                {
                    possibleRotations.Add(0.0f);
                }
            }

            bool hasAdjacentRoadNeighbour = !(possibleRotations.Count == 0);    //Check to see if we have a road neighbour

            if (possibleRotations.Count == 0)
            {
                //We don't have any neighbours so let's just pick a random rotation
                possibleRotations.AddRange(new List<float>() { 0.0f, 90.0f, 180.0f, 270.0f });
            }

            //Decide upon a rotation
            Randomizer.Regenerate();
            int index = Randomizer.RNG.Next(0, possibleRotations.Count - 1);

            //Compute the chunk in which the building resides
            int roundedBuildingX = Convert.ToInt32(Math.Floor((float)tilePos.x / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize) * ConfigurationManager.Instance.Missions.MissionSpawnChunkSize);
            int roundedBuildingY = Convert.ToInt32(Math.Floor((float)tilePos.y / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize) * ConfigurationManager.Instance.Missions.MissionSpawnChunkSize);
            int buildingHorizontalChunk = roundedBuildingX / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize;
            int buildingVerticalChunk = roundedBuildingY / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize;

            //Create our tile info based on our computations
            _BuildingTiles[tilePos] = new BuildingTileInfo(tilePos, hasAdjacentRoadNeighbour, isInternal, buildingHorizontalChunk, buildingVerticalChunk);

            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when adding new building tile, returning false. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Identifies all possible building tiles in the city
    /// </summary>
    /// <returns>Were the tiles identified successfully?</returns>
    private bool IdentifyBuildingTiles()
    {
        try
        {
            //Create our tile infos, compute the position boundaries of buildings
            _BuildingTiles = new Dictionary<Vector3Int, BuildingTileInfo>();

            int margin = ConfigurationManager.Instance.Core.CityBoxFillMargin;
            BuildingAreaBottomLeftTilePosition = new Vector3Int(GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.x - margin,
                                                                GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - margin,
                                                                GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.z);

            BuildingAreaTopLeftTilePosition = new Vector3Int(GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.x - margin,
                                                                GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.y + margin,
                                                                GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.z);

            BuildingAreaBottomRightTilePosition = new Vector3Int(GameController.Instance.LSystem.Drawer.CityBottomRightTilePosition.x + margin,
                                                                GameController.Instance.LSystem.Drawer.CityBottomRightTilePosition.y - margin,
                                                                GameController.Instance.LSystem.Drawer.CityBottomRightTilePosition.z);


            BuildingAreaTopRightTilePosition = new Vector3Int(GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x + margin,
                                                                GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.y + margin,
                                                                GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.z);

            BuildingAreaSize = new Vector2Int(BuildingAreaTopRightTilePosition.x - BuildingAreaTopLeftTilePosition.x, BuildingAreaTopLeftTilePosition.y - BuildingAreaBottomLeftTilePosition.y);


            Vector3Int startPos = new Vector3Int(GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.x - margin,
                                                    GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - margin,
                                                    GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.z
                                                    );

            //First, fill up all the exposed outside sections of the city within the boundaries using a box fill
            GameManager.Instance.RoadsTilemap.BoxFill(BuildingAreaBottomLeftTilePosition,
                                    Resources.Load("Palette Tiles/red") as Tile,
                                    GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.x - margin,
                                    GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - margin,
                                    GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x + margin,
                                    GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.y + margin
                                    );

            //Now we loop through for the internal sections
            for (int x = GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.x - margin; x <= GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x + margin; x++)
            {
                for (int y = GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.y + margin; y >= GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - margin; y--)
                {
                    Vector3Int thisTilePos = new Vector3Int(x, y, Constants.TilemapZPosition);
                    GameManager.Instance.BuildingsTilemap.SetTile(thisTilePos, _TransparentTile);

                    TileBase thisBuildingTile = GameManager.Instance.BuildingsTilemap.GetTile(thisTilePos);
                    TileBase thisRoadTile = GameManager.Instance.RoadsTilemap.GetTile(thisTilePos);

                    if (thisRoadTile != null && thisRoadTile.name.Contains("red"))
                    {
                        //This tile got filled so we can place a building here
                        if (!AddNewBuildingTile(thisTilePos, false))
                        {
                            Debug.LogError("ERROR: Failed to add new filled building tile at position " + thisTilePos + ", returning false.");
                            return false;
                        }
                    }

                    if (thisRoadTile == null || (!thisRoadTile.name.Contains("red") && !thisRoadTile.name.Contains("Road")))
                    {
                        //This tile didn't get filled, but it's not a road either so it must be an internal section
                        if (!AddNewBuildingTile(thisTilePos, true))
                        {
                            Debug.LogError("ERROR: Failed to add new un-filled building tile at position " + thisTilePos + ", returning false.");
                            return false;
                        }
                    }
                }
            }

            //Now let's tidy up from the fill
            GameManager.Instance.RoadsTilemap.BoxFill(startPos,
                                    Resources.Load("Palette Tiles/transparent") as Tile,
                                    GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.x - margin,
                                    GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - margin,
                                    GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x + margin,
                                    GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.y + margin
                                    );

            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when identifying building tiles, returning false. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Gets the next key building tile info that is unused
    /// </summary>
    /// <returns>The building tile info for the tile if successful, otherwise null</returns>
    private BuildingTileInfo GetNextKeyBuildingTileInfo()
    {
        List<BuildingTileInfo> tileInfos = _BuildingTiles.Where(x => x.Value.IsAdjacentToRoad == true).Select(x => x.Value).ToList();   //Get all road adjacent building tiles

        while (true)
        {
            Randomizer.Regenerate();
            int index = Randomizer.RNG.Next(0, tileInfos.Count - 1);

            if (!_UsedKeyIndexes.Contains(index))
            {
                //Found one we can use, return it
                _UsedKeyIndexes.Add(index);
                return tileInfos[index];
            }

            else
            {
                if (_UsedKeyIndexes.Count >= _BuildingTiles.Count)
                {
                    //No tiles left
                    return null;
                }
            }
        }
    }

    /// <summary>
    /// Sets up the colliders for a building model
    /// </summary>
    /// <param name="buildingGameObject">The model to set up the collisions for</param>
    /// <returns>Was the setup successful?</returns>
    private bool SetupColliders(GameObject buildingGameObject)
    {
        try
        {
            //Get the mesh, round the size to the nearest whole unit
            Mesh mesh = buildingGameObject.GetComponent<MeshFilter>().mesh;

            int roundedWidth = Convert.ToInt32(Math.Round(mesh.bounds.size.x, 0, MidpointRounding.AwayFromZero));
            int roundedHeight = Convert.ToInt32(Math.Round(mesh.bounds.size.y, 0, MidpointRounding.AwayFromZero));

            if (roundedWidth < 1)
            {
                roundedWidth = 1;
            }

            if (roundedHeight < 1)
            {
                roundedHeight = 1;
            }

            //Set up the edge colliders based on the rounded unit sizes
            buildingGameObject.FindChild("BuildingTopCollider").transform.localScale = new Vector3(roundedWidth, 1.0f, 1.0f);
            buildingGameObject.FindChild("BuildingTopCollider").transform.localPosition = new Vector3(0.0f, -(roundedHeight / 2.0f) - 0.5f, 0.0f);

            buildingGameObject.FindChild("BuildingBottomCollider").transform.localScale = new Vector3(roundedWidth, 1.0f, 1.0f);
            buildingGameObject.FindChild("BuildingBottomCollider").transform.localPosition = new Vector3(0.0f, (roundedHeight / 2.0f) + 0.5f, 0.0f);

            buildingGameObject.FindChild("BuildingLeftCollider").transform.localScale = new Vector3(1.0f, roundedHeight, 1.0f);
            buildingGameObject.FindChild("BuildingLeftCollider").transform.localPosition = new Vector3(-(roundedWidth / 2.0f) - 0.5f, 0.0f, 0.0f);

            buildingGameObject.FindChild("BuildingRightCollider").transform.localScale = new Vector3(1.0f, roundedHeight, 1.0f);
            buildingGameObject.FindChild("BuildingRightCollider").transform.localPosition = new Vector3((roundedWidth / 2.0f) + 0.5f, 0.0f, 0.0f);

            return true;
        }

        catch(Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when setting up building colliders. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Tries to set which tiles are occupied for the building model on the tile
    /// </summary>
    /// <param name="tileInfo">The tile info for the tile the building is on</param>
    /// <param name="modelMesh">The building model</param>
    /// <param name="rotation">The rotation of the building</param>
    /// <returns>Was the set successful?</returns>
    private bool TrySetOccupiedTiles(BuildingTileInfo tileInfo, Mesh modelMesh, float rotation)
    {
        List<Vector3Int> tilePositions = new List<Vector3Int>();

        //Compute the rounded size of the building model
        Vector2 roundedSize = new Vector2((float)Math.Ceiling(modelMesh.bounds.size.x), (float)Math.Ceiling(modelMesh.bounds.size.y));
        rotation = Mathf.Abs(rotation);

        if (roundedSize.x != roundedSize.y)
        {
            if (rotation == 90.0f || rotation == 270.0f)
            {
                //Flip the size around to account for the rotation
                float x = roundedSize.x;
                float y = roundedSize.y;
                roundedSize = new Vector2(y, x);
            }
        }

        for (int x = 0; x < roundedSize.x; x++)
        {
            for (int y = 0; y < roundedSize.y; y++)
            {
                //Now loop through the size, get the tile based on an offset from the tile info position
                Vector3Int thisPosition = new Vector3Int(tileInfo.Position.x + x, tileInfo.Position.y + y, tileInfo.Position.z);

                TileBase thisBuildingTile = GameManager.Instance.BuildingsTilemap.GetTile(thisPosition);
                TileBase thisRoadTile = GameManager.Instance.RoadsTilemap.GetTile(thisPosition);

                if ((thisBuildingTile != null && thisBuildingTile.name == Constants.OccupiedBuildingTileID) || (thisRoadTile != null && thisRoadTile.name.Contains("Road")))
                {
                    //The tile is occupied so we can't place it
                    return false;
                }

                else
                {
                    //Let's occupy this tile
                    tilePositions.Add(thisPosition);
                }
            }
        }

        foreach (Vector3Int tilePosition in tilePositions)
        {
            if (GameManager.Instance.BuildingsTilemap.GetTile(tilePosition) != null)
            {
                //Mark it at occupied
                GameManager.Instance.BuildingsTilemap.GetTile(tilePosition).name = Constants.OccupiedBuildingTileID;
            }

            else
            {
                //Failed to get the tile
                return false;
            }
        }

        return true;
    }

    /// <summary>
    /// Spawns a garage with the ID and type at a random position
    /// </summary>
    /// <param name="buildingID">The ID of the buillding object</param>
    /// <param name="type">The type of garage to spawn</param>
    /// <returns>Was the spawn successful?</returns>
    private bool SpawnGarage(string buildingID, Constants.CarType type)
    {
        try
        {
            BuildingTileInfo tileInfo = GetNextKeyBuildingTileInfo();   //Let's get a key building tile info
            if (tileInfo != null)
            {
                //Got one, get the building instance and model from the ID
                TileBase tile = GameManager.Instance.BuildingsTilemap.GetTile(tileInfo.Position);
                tile.name = Constants.OccupiedBuildingTileID;

                BuildingInstance buildingInstance = BuildingsInfo.Buildings[buildingID];
                BuildingModel buildingModel = BuildingsInfo.BuildingModels[buildingInstance.ModelID];

                //Set up the game object itself
                GameObject modelObject = Resources.Load("Buildings/" + buildingModel.Path) as GameObject;
                Mesh modelMesh = modelObject.GetComponent<MeshFilter>().sharedMesh;
                Material modelMaterial = modelObject.GetComponent<MeshRenderer>().sharedMaterial;

                //Now instantiate from the prefab
                GameObject garageGameObject = Instantiate(GameController.Instance.GarageBuildingPrefab);
                garageGameObject.GetComponent<MeshFilter>().mesh = modelMesh;
                garageGameObject.GetComponent<MeshRenderer>().sharedMaterial = modelMaterial;

                //Set up the rotation and parent
                garageGameObject.GetComponent<GarageController>().Type = type;
                garageGameObject.transform.eulerAngles = new Vector3(180.0f, 0.0f, 0.0f);
                garageGameObject.transform.parent = GameManager.Instance.BuildingsTilemap.transform;

                //Now sample Perlin Noise to displace the height
                int noiseValue = Convert.ToInt32(Math.Round(_PerlinNoise.Samples[tileInfo.Position.x][tileInfo.Position.y].r * ConfigurationManager.Instance.Generation.SkyscraperPerlinMultiplier));
                float rotation = Utilities.GetRandomNESWRotation();

                garageGameObject.transform.position = Utilities.GetModelWorldPosition(tileInfo.Position, modelMesh, rotation);

                //We only displace if the model is a Skyscraper or Commercial
                if (buildingModel is SkyscraperModel)
                {
                    garageGameObject.transform.position = new Vector3
                    (
                        garageGameObject.transform.position.x,
                        garageGameObject.transform.position.y,
                        garageGameObject.transform.position.z + ((float)noiseValue * ((SkyscraperModel)buildingModel).StoreyHeightMultiplier)
                    );
                }

                else if (buildingModel is CommercialModel)
                {
                    garageGameObject.transform.position = new Vector3
                    (
                        garageGameObject.transform.position.x,
                        garageGameObject.transform.position.y,
                        garageGameObject.transform.position.z + ((float)noiseValue * ((CommercialModel)buildingModel).StoreyHeightMultiplier)
                    );
                }

                //Depending on the garage type, change the icon colour
                Color garageColour = Color.white;

                switch(type)
                {
                    case Constants.CarType.LowEnd:
                        garageColour = ColoursManager.Instance.Colours["LowEndGarageIcon"].Colour;
                        break;

                    case Constants.CarType.MediumEnd:
                        garageColour = ColoursManager.Instance.Colours["MediumEndGarageIcon"].Colour;
                        break;

                    case Constants.CarType.HighEnd:
                        garageColour = ColoursManager.Instance.Colours["HighEndGarageIcon"].Colour;
                        break;
                }

                //Add the icon and store the garage data
                MinimapManager.Instance.AddIcon(garageGameObject, buildingID, garageColour, ConfigurationManager.Instance.Minimap.GarageIconRenderSize, ((int)MinimapIconFlags.NotOnMission), Resources.Load<Sprite>("Minimap/Icons/minimapGarageIcon"));

                Garages[tileInfo.Position] = tileInfo;

                return true;
            }

            else
            {
                //TODO: We've used up all our spots, this is extremely unlikely to happen but we may want to handle this explicitly in the future
            }

            return false;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when spawning garage. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Tries to spawn a random skyscraper model using the tile info
    /// </summary>
    /// <param name="tileInfo">The info of the tile to spawn the skyscraper on</param>
    /// <returns>Was the spawn successful?</returns>
    private bool TrySpawnRandomSkyscraper(BuildingTileInfo tileInfo)
    {
        try
        {
            //Get a random model and rotation
            List<SkyscraperModel> skyscrapers = BuildingsInfo.BuildingModels.Values.Where(x => x.GetType() == typeof(SkyscraperModel)).Cast<SkyscraperModel>().ToList();
            Randomizer.Regenerate();
            int index = Randomizer.RNG.Next(0, skyscrapers.Count);
            SkyscraperModel skyscraperModel = skyscrapers[index];
            float rotation = Utilities.GetRandomNESWRotation();

            //Setup the game object
            GameObject modelObject = Resources.Load("Buildings/" + skyscraperModel.Path) as GameObject;
            Mesh modelMesh = modelObject.GetComponent<MeshFilter>().sharedMesh;
            Material modelMaterial = modelObject.GetComponent<MeshRenderer>().sharedMaterial;

            if (TrySetOccupiedTiles(tileInfo, modelMesh, rotation))
            {
                //Everything has been set to be occupied, set the object properties
                GameObject buildingGameObject = Instantiate(GameController.Instance.DecorativeBuildingPrefab);
                buildingGameObject.GetComponent<MeshFilter>().mesh = modelMesh;
                buildingGameObject.GetComponent<MeshRenderer>().sharedMaterial = modelMaterial;
                buildingGameObject.transform.eulerAngles = new Vector3(180.0f, 0.0f, rotation);
                buildingGameObject.transform.parent = GameManager.Instance.BuildingsTilemap.transform;

                //Displace based on Perlin Noise
                int noiseValue = Convert.ToInt32(Math.Round(_PerlinNoise.Samples[tileInfo.Position.x][tileInfo.Position.y].r * ConfigurationManager.Instance.Generation.SkyscraperPerlinMultiplier));

                buildingGameObject.transform.position = Utilities.GetModelWorldPosition(tileInfo.Position, modelMesh, rotation);

                buildingGameObject.transform.position = new Vector3
                (
                    buildingGameObject.transform.position.x,
                    buildingGameObject.transform.position.y,
                    buildingGameObject.transform.position.z + ((float)noiseValue * skyscraperModel.StoreyHeightMultiplier)
                );

                if(SetupColliders(buildingGameObject))
                {
                    //Colliders set up, set occupied
                    tileInfo.SetOccupied(buildingGameObject);
                    return true;
                }

                else
                {
                    //Failed to set up colliders, let's abort
                    Debug.LogError("ERROR: Failed to setup colliders for building, destroying...");
                    Destroy(buildingGameObject);
                }
            }

            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when spawning random skyscraper. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Tries to spawn a random commercial model using the tile info
    /// </summary>
    /// <param name="tileInfo">The info of the tile to spawn the commercial building on</param>
    /// <returns>Was the spawn successful?</returns>
    private bool TrySpawnRandomCommercial(BuildingTileInfo tileInfo)
    {
        try
        {
            //Get a random model and rotation
            List<CommercialModel> commercials = BuildingsInfo.BuildingModels.Values.Where(x => x.GetType() == typeof(CommercialModel)).Cast<CommercialModel>().ToList();
            Randomizer.Regenerate();
            int index = Randomizer.RNG.Next(0, commercials.Count);
            CommercialModel commercialModel = commercials[index];
            float rotation = Utilities.GetRandomNESWRotation();

            //Setup the game object
            GameObject modelObject = Resources.Load("Buildings/" + commercialModel.Path) as GameObject;
            Mesh modelMesh = modelObject.GetComponent<MeshFilter>().sharedMesh;
            Material modelMaterial = modelObject.GetComponent<MeshRenderer>().sharedMaterial;

            if (TrySetOccupiedTiles(tileInfo, modelMesh, rotation))
            {
                //Everything has been set to be occupied, set the object properties
                GameObject buildingGameObject = Instantiate(GameController.Instance.DecorativeBuildingPrefab);
                buildingGameObject.GetComponent<MeshFilter>().mesh = modelMesh;
                buildingGameObject.GetComponent<MeshRenderer>().sharedMaterial = modelMaterial;
                buildingGameObject.transform.eulerAngles = new Vector3(180.0f, 0.0f, rotation);
                buildingGameObject.transform.parent = GameManager.Instance.BuildingsTilemap.transform;

                //Displace based on Perlin Noise
                int noiseValue = Convert.ToInt32(Math.Round(_PerlinNoise.Samples[tileInfo.Position.x][tileInfo.Position.y].r * ConfigurationManager.Instance.Generation.CommercialPerlinMultiplier));

                buildingGameObject.transform.position = Utilities.GetModelWorldPosition(tileInfo.Position, modelMesh, rotation);

                buildingGameObject.transform.position = new Vector3
                (
                    buildingGameObject.transform.position.x,
                    buildingGameObject.transform.position.y,
                    buildingGameObject.transform.position.z + ((float)noiseValue * commercialModel.StoreyHeightMultiplier)
                );

                if (SetupColliders(buildingGameObject))
                {
                    //Colliders set up, set occupied
                    tileInfo.SetOccupied(buildingGameObject);
                    return true;
                }

                else
                {
                    //Failed to set up colliders, let's abort
                    Debug.LogError("ERROR: Failed to setup colliders for building, destroying...");
                    Destroy(buildingGameObject);
                }
            }

            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when spawning random commercial. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Tries to spawn a random house model using the tile info
    /// </summary>
    /// <param name="tileInfo">The info of the tile to spawn the house on</param>
    /// <returns>Was the spawn successful?</returns>
    private bool TrySpawnRandomHouse(BuildingTileInfo tileInfo)
    {
        try
        {
            //Get a random model and rotation
            List<HouseModel> houses = BuildingsInfo.BuildingModels.Values.Where(x => x.GetType() == typeof(HouseModel)).Cast<HouseModel>().ToList();
            Randomizer.Regenerate();
            int index = Randomizer.RNG.Next(0, houses.Count);
            HouseModel houseModel = houses[index];
            float rotation = Utilities.GetRandomNESWRotation();

            //Setup the game object
            GameObject modelObject = Resources.Load("Buildings/" + houseModel.Path) as GameObject;
            Mesh modelMesh = modelObject.GetComponent<MeshFilter>().sharedMesh;
            Material modelMaterial = modelObject.GetComponent<MeshRenderer>().sharedMaterial;

            if (TrySetOccupiedTiles(tileInfo, modelMesh, rotation))
            {
                //Everything has been set to be occupied, set the object properties
                GameObject buildingGameObject = Instantiate(GameController.Instance.DecorativeBuildingPrefab);
                buildingGameObject.GetComponent<MeshFilter>().mesh = modelMesh;
                buildingGameObject.GetComponent<MeshRenderer>().sharedMaterial = modelMaterial;
                buildingGameObject.transform.eulerAngles = new Vector3(180.0f, 0.0f, rotation);
                buildingGameObject.transform.parent = GameManager.Instance.BuildingsTilemap.transform;
                buildingGameObject.transform.position = Utilities.GetModelWorldPosition(tileInfo.Position, modelMesh, rotation);

                if (SetupColliders(buildingGameObject))
                {
                    //Colliders set up, set occupied
                    tileInfo.SetOccupied(buildingGameObject);
                    return true;
                }

                else
                {
                    //Failed to set up colliders, let's abort
                    Debug.LogError("ERROR: Failed to setup colliders for building, destroying...");
                    Destroy(buildingGameObject);
                }
            }

            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when spawning random house. The exception is: " + ex);
            return false;
        }
    }

    /// <summary>
    /// Determines whether a building should spawn in the noise building type transition zone or not
    /// </summary>
    /// <param name="tileInfo">The tile info of the tile to spawn on</param>
    /// <returns>Should a spawn occur?</returns>
    private bool ShouldSpawnInZone(BuildingTileInfo tileInfo)
    {
        return _PerlinNoise.Samples[tileInfo.Position.x][tileInfo.Position.y].r <=  ConfigurationManager.Instance.Generation.BuildingTypeNoiseZonePerlinCap;
    }

    /// <summary>
    /// Tries to spawn a building on the tile represented by the tile info
    /// </summary>
    /// <param name="tileInfo">The tile info of the tile to spawn on</param>
    /// <param name="mustSpawn">Must a building spawn here?</param>
    /// <returns>Was the spawn successful?</returns>
    private bool SpawnBuilding(BuildingTileInfo tileInfo, bool mustSpawn = false)
    {
        //Compute the distance from the city centre
        float distFromCentre = Math.Abs(Mathf.Sqrt(Mathf.Pow(GameController.Instance.LSystem.Drawer.CityCentreTilePosition.x - tileInfo.Position.x, 2.0f) +
                                                Mathf.Pow(GameController.Instance.LSystem.Drawer.CityCentreTilePosition.y - tileInfo.Position.y, 2.0f)));

        //Near the centre, spawn a skyscraper
        if (distFromCentre <= _SkyscraperDistance)
        {
            return TrySpawnRandomSkyscraper(tileInfo);
        }

        //In the noise zone, spawn either a skyscraper or a commercial building
        else if (distFromCentre <= _SkyscraperDistance + _NoiseZoneDistance)
        {
            if (ShouldSpawnInZone(tileInfo) || mustSpawn)
            {
                return TrySpawnRandomSkyscraper(tileInfo);
            }

            else
            {
                return TrySpawnRandomCommercial(tileInfo);
            }
        }

        //Mid-city, spawn a commercial building
        else if (distFromCentre <= _CommercialDistance && !mustSpawn)
        {
            return TrySpawnRandomCommercial(tileInfo);
        }

        //In the noise zone, spawn either a commercial building or a house
        else if (distFromCentre <= _CommercialDistance + _NoiseZoneDistance)
        {
            if (ShouldSpawnInZone(tileInfo))
            {
                return TrySpawnRandomCommercial(tileInfo);
            }

            else
            {
                return TrySpawnRandomHouse(tileInfo);
            }
        }

        //Outskirts of the city, spawn a house
        else
        {
            return TrySpawnRandomHouse(tileInfo);
        }
    }

    /// <summary>
    /// Spawns key important buildings into the city
    /// </summary>
    /// <returns>Was the placement successful?</returns>
    private bool PlaceKeyBuildings()
    {
        //Spawn a garage for each car class
        if (SpawnGarage("LowEndGarage", Constants.CarType.LowEnd))
        {
            if (SpawnGarage("MediumEndGarage", Constants.CarType.MediumEnd))
            {
                if (SpawnGarage("HighEndGarage", Constants.CarType.HighEnd))
                {
                    return true;    //All garages spawned successfully!
                }

                else
                {
                    Debug.LogError("ERROR: Failed to spawn high-end garage.");
                }
            }

            else
            {
                Debug.LogError("ERROR: Failed to spawn medium-end garage.");
            }
        }

        else
        {
            Debug.LogError("ERROR: Failed to spawn low-end garage.");
        }

        return false;   //We must have failed to spawn a garage
    }

    /// <summary>
    /// Places decorative buildings around the city
    /// </summary>
    /// <returns>Was the placement successful?</returns>
    private bool PlaceDecorativeBuildings()
    {
        try
        {
            List<BuildingTileInfo> tileInfos = _BuildingTiles.Values.ToList();

            foreach (BuildingTileInfo tileInfo in tileInfos)
            {
                TileBase thisTile = GameManager.Instance.BuildingsTilemap.GetTile(tileInfo.Position);

                if (thisTile.name.Contains(Constants.ValidBuildingPlacementTileID))
                {
                    //City outskirts adjacent to road, MUST have a buidling
                    if (tileInfo.IsAdjacentToRoad && !tileInfo.IsInternal)
                    {
                        SpawnBuilding(tileInfo, true);
                    }

                    else
                    {
                        //Use the randomizer and the density to determine whether a building should spawn or not
                        Randomizer.Regenerate();
                        if (Randomizer.RNG.Next(0, 100) <= GameController.Instance.CityDensity)
                        {
                            SpawnBuilding(tileInfo);
                        }
                    }
                }
            }

            return true;
        }

        catch(Exception ex)
        {
            return false;
        }
    }

    /// <summary>
    /// Core function for all building placements in the city
    /// </summary>
    /// <returns>Were buildings placed successfully?</returns>
    public bool PlaceBuildings()
    {
        try
        {
            //Compute the type distances based on the configuration data
            _SkyscraperDistance = Math.Abs(Mathf.Sqrt(Mathf.Pow(((GameController.Instance.LSystem.Drawer.CitySize.x / 100.0f) * GameController.Instance.SkyscrapersZoneSizePercentage), 2.0f) +
                                    Mathf.Pow(((GameController.Instance.LSystem.Drawer.CitySize.y / 100.0f) * GameController.Instance.SkyscrapersZoneSizePercentage), 2.0f))) / 2.0f;


            _CommercialDistance = Math.Abs(Mathf.Sqrt(Mathf.Pow(((GameController.Instance.LSystem.Drawer.CitySize.x / 100.0f) * GameController.Instance.CommercialZoneSizePercentage), 2.0f) +
                                    Mathf.Pow(((GameController.Instance.LSystem.Drawer.CitySize.y / 100.0f) * GameController.Instance.CommercialZoneSizePercentage), 2.0f))) / (2.0f) + _SkyscraperDistance;

            _NoiseZoneDistance = ConfigurationManager.Instance.Generation.BuildingTypeNoiseZoneSize;

            //We reset these here to maintain determinism over several runs
            RoadBuildings = new Dictionary<Vector3Int, BuildingTileInfo>();
            Garages = new Dictionary<Vector3Int, BuildingTileInfo>();
            _UsedKeyIndexes = new List<int>();

            if (IdentifyBuildingTiles())
            {
                //Got our tiles, let's create the Perlin Noise generator
                _PerlinNoise = new PerlinNoiseGenerator(new Vector2Int(GameManager.Instance.BuildingsTilemap.size.x, GameManager.Instance.BuildingsTilemap.size.y), 35.0f, new Vector2Int(1, 1), GameController.Instance.PerlinOffset);
                _PerlinNoise.Generate();

                if (PlaceKeyBuildings())
                {
                    //Our key buildings are placed, now let's decorate
                    if (PlaceDecorativeBuildings())
                    {
                        //The city is decorated, let's store all the buildings beside roads for mission spawning
                        foreach(KeyValuePair<Vector3Int, BuildingTileInfo> tileKeyVal in _BuildingTiles)
                        {
                            if(tileKeyVal.Value.IsAdjacentToRoad && tileKeyVal.Value.IsOccupied)
                            {
                                RoadBuildings[tileKeyVal.Key] = tileKeyVal.Value;
                            }
                        }

                        return true;
                    }
                }
            }

            return false;   //Something went wrong
        }

        catch
        {
            return false;
        }
    }

    /// <summary>
    /// Clears the entire building tilemap
    /// </summary>
    public void Clear()
    {
        try
        {
            for (int x = 0; x < GameManager.Instance.BuildingsTilemap.size.x; x++)
            {
                for (int y = 0; y < GameManager.Instance.BuildingsTilemap.size.y; y++)
                {
                    Vector3Int thisTilePos = new Vector3Int(x, y, Constants.TilemapZPosition);
                    GameManager.Instance.BuildingsTilemap.SetTile(thisTilePos, _TransparentTile);
                }
            }
        }

        catch { }
    }
}
